package G;
use strict;

our $VERSION;
my ($usesWin, $parseCmdline, $noexec);
BEGIN {
	$VERSION = '4.1.2.0';

	$usesWin = ($^O eq 'MSWin32' ? 1 : 0);
	$parseCmdline
		= (
				(
					$usesWin and $ENV{COMSPEC} =~ m{[/\\](?:4nt\.exe|cmd\.exe|command\.com)$}i # using a shell that requires it
				or
					(caller())[2] == 0 # -MG in parameters or in PERL5OPT
				)
			?	1
			:	0
			);
	$noexec = 0;

	if ($usesWin) {
		eval "use File::DosGlob qw(glob);"
	}

}

my %char = (
	'n' => "\n",
	'f' => "\f",
	' ' => ' ',
	't' => "\t",
	'r' => "\r",
	'a' => "\a",
);

sub bslash ($) {
    $_[0] =~ s/\\(.)/$char{$1}||"\\$1"/ge;
}

sub mask2re {
	my $mask = '^'.shift().'$';
	$mask =~ s/([\.\(\)\[\]\+\\])/\\$1/g;
	$mask =~ s/\*/.*/g;
	$mask =~ tr/?/./;
	return qr/$mask/i;
}

sub ParseCommandLine {
	local $_ = shift;
	my @argv;
#print "splitting command line : $_\n\n";
	$_ =~ s/^\s+//;
	$_ =~ s/\\\\/\x0/g; #escape the double backslashes
	while ($_ ne '') {
		if (s/^''// or s/^""// or s/^``//) { # empty
			push @argv, '';

		} elsif (s/^'(.*?[^\\])'//) { # singlequoted
			my $arg = $1;
			$arg =~ s/\\'/'/g;
			$arg =~ s/\\ / /g;
			$arg =~ s/\x0/\\\\/g; # \\server\share !!!
			push @argv, $arg;

		} elsif (s/^`(.*?[^\\])`//){ # backticks
			my $arg = $1;
			$arg =~ s/\\`/`/g;
			bslash $arg if $arg =~ s/^://;
			$arg =~ s/\x0/\\\\/g;
			if ($noexec) {
				push @argv, $arg
			} else {
				push @argv, scalar(`$arg`);
			}

		} elsif (s/^"(.*?[^\\])"// or s/^(\S+)//) {
			my $arg = $1;
			$arg =~ s/\\"/"/g;
			if (! $usesWin) {
				bslash $arg;
				$arg =~ s/\x0/\\\\/g;
				push @argv, glob $arg;

			} elsif ($arg =~ s/^://) {
				bslash $arg;
				$arg =~ s/\x0/\\/g;
				push @argv, $arg;

			} elsif ($arg =~ /(^|[^\\])[*?]/) {
				$arg =~ s/\x0/\\/g;
				my @glob = glob $arg;
				if (@glob) {
					push @argv, @glob;
				} else {
					push @argv, $arg;
				}

			} else {
				$arg =~ s/\\([?*])/$1/g;
				$arg =~ s/\x0/\\\\/g;
				push @argv, $arg;
			}

		} else {
			die <<"*END*";
Sorry, this should never ever happen! Please contact Jenda\@Krynicky.cz
and report an error in G.pm ver. $VERSION with command line
	$G::cmdline\n
*END*
		}

		$_ =~ s/^\s+//;
	}
	return @argv;
}

sub StripPerlFromCommandLine { # read the command line and get rid of perl.exe or similar stuff from there.
	local $_ = shift();
	if ($0 eq '-e' or $0 eq '"-e"') { # perl -e "code" params
		s/^.*? "?-e"?\s*/-e /;
		s/^""// or s/^''// or s/^".*?[^\\]"// or s/^'.*?[^\\]'// or s/^\S+//
			while s/^\s*"?-e"?\s*//;
		s/^\s+//;
	} elsif (!(-e $INC{'G.pm'})) { # compiled_script.exe params
		s/^".*?"\s*// or s/^\S+\s*//; # strip the executable
	} else { # perl script.pl params
		s/^".*?"\s*// or s/^\S+\s*//; # strip the perl executable
		1 while (s/^(-[^e]".*?"|-[^e]\S*|"-[^e].*?")\s*//); # strip parameters
		s/^".*?"\s*// or s/^\S+\s*//; # strip the script name
	}
	return $_;
}

BEGIN {
	if ($usesWin) {
		local $/;
		eval <<'*END_CODE*';
			require DynaLoader;

			our @ISA = qw(DynaLoader);
			bootstrap G $VERSION; # defines GetCommandLine()

			$G::cmdline = StripPerlFromCommandLine(GetCommandLine());
*END_CODE*
		die $@,"\n" if $@;
	} else { # for other OSes
		eval <<'*END_CODE*';
			if ($parseCmdline) {
				$G::cmdline = q{'}.join(q{' '}, map {s/['\\]/\\$1/g} @ARGV).q{'};
			} else {
				$G::cmdline = q{"}.join(q{" "}, map {s/["\\]/\\$1/g} @ARGV).q{"};
			}
			sub GetCommandLine {
				if ($0 eq '-e') {
					return "$^X -e '' $G::cmdline";
				} else {
					return "$^X '$0' $G::cmdline";
				}
			}
*END_CODE*
			die $@,"\n" if $@;
	}
}

sub import {
	my $caller_pack = caller;
	my $noglob = 0;
	foreach (@_) {
		if ($_ =~ /GetCommandLine|ParseCommandLine|StripPerlFromCommandLine/) {
			no strict 'refs';
			*{$caller_pack."::".$_} = \&{$_};
		} elsif ($_ =~ /^:?NOGLOB/i) {
			$noglob = 1;
		} elsif ($_ =~ /^:?NOEXEC/i) {
			$noexec = 1;
		} elsif ($_ =~ /^:?R/i) {
			local $^W;
			eval <<'*END*';
			use File::Find;
			sub glob {
				local $_;
				my $mask = shift();
				my @files;
				if ($mask =~ m{^(.*)[/\\](.*)$}) {
					my $dir = $1;
					my $re = mask2re($2);
					File::Find::find( sub {push @files, $File::Find::name if /$re/}, $dir);
				} else {
					my $re = mask2re($mask);
					File::Find::find( sub {push @files, $File::Find::name if /$re/}, '.');
				}
				return @files;
			}
*END*
		}
	}
	return 1 if $noglob or !$parseCmdline;
	return 1 if $G::alreadyglobed++;
	@ARGV = ParseCommandLine( $G::cmdline);
}

sub unimport {
	$G::alreadyglobed++;
}

1;

__END__

=head1 NAME

G - command line parameter globing (mainly for Win32)

version 4.1.2.0

=head1 SYNOPSIS

	use G;

	use G qw(NOEXEC GetCommandLine);

	perl -MG=NOEXEC script.pl *.txt 'foo bar baz'

	SET PERL5OPT=-MG
	perl script.pl *.txt 'foo bar baz'

=head1 DESCRIPTION

The module will take the command line and parse it as the God intended,
taking into account both double and single quotes
(not entirely, but close to the Unix way) and even backticks.

It was originaly meant to work only under Windows, but the functions are now
exported and may be used even under Unix and other non-Windows systems.

You may either C<use G> it in your scripts (in this case the globbing will be done only
if the script runs under Windows and the $ENV{COMSPEC} is cmd.exe, command.com or
4nt.exe) or add c<-MG> into the command line or $ENV{PERL5OPT} (in this case the globbing
will be done even under Unix).

This allows you to use the list version of system() and still get the parameters globed :

	system( 'perl', '-MG=NOEXEC', 'script.pl', @params);

=head2 Examples

(For windows)

The module processes (since version 3.0) the plain command line, not the
list broken (in both meanings) by C runtime it understands single quotes
and backticks as well.

  perl -MG -e "print join(qq{\n}, @ARGV)" *.txt
 versus
  perl -MG -e "print join(qq{\n}, @ARGV)" '*.txt'


  perl -MG -e "print join(qq{\n}, @ARGV)" 'hello world'
  perl -MG -e "print join(qq{\n}, @ARGV)" 'hello   world'

The number of spaces will be preserved !

You may even use the backslash escapes in doublequoted or unquoted
parameters, but since windoze use backslash as the directory separator,
the module does the escape interpolation only if you begin the parameter by ':'.

  perl -MG -e "print '<',join(qq{>\n<}, @ARGV),'>'" "hello \n world"
 versus
  perl -MG -e "print '<',join(qq{>\n<}, @ARGV),'>'" ":hello \n world"

The only characters that will be escaped in other unquoted or
doublequoted strings are doublequotes, wildchars (* and ?) and backslashes :

  perl -MG -e "print '<',join(qq{>\n<}, @ARGV),'>'" "I said: \"How are you\?\""

in single quoted it's only the single quotes and backslashes

  perl -MG -e "print '<',join(qq{>\n<}, @ARGV),'>'" 'Hi d\'Artagnan'

backslashes folowed by other characters are preserved!

The other cool feature is the backticks:

 perl -MG -e "print '<',join(qq{>\n<}, @ARGV),'>'" `dir c:\\`

Please note that you have to double the backslash before the closing backtick !

Please note also that

 perl -MG -e "print '<',join(qq{>\n<}, @ARGV),'>'" "hello"'world'

will give you TWO arguments, not one !

Enjoy ;-)

=head2 Options

=head3 NOEXEC

"Do not execute". Tells the module to treat backticks like doublequotes.

	perl -MG -e "print join(qq{\n}, @ARGV)" `dir /b .`

returns the list of files in current directory, while

	perl -MG=NOEXEC -e "print join(qq{\n}, @ARGV)" `dir /b .`

returns c<dir /b .>.

You may use it either as

	perl -MG=NOEXEC ...

or

	use G qw(NOEXEC ...);

=head3 NOGLOB

"Do not do the globing". Allows you to import the functions without modifying the @ARGV.

	use G qw(NOGLOB GetCommandLine);

=head3 RECURSIVE or R

Do the globbing recursively.

	perl -MG=R -e "print join(qq{\n}, @ARGV)" *.txt

Please note that you can ONLY use a mask for the file in this case! If you want to find
all *.txt files under a certain dir you can use this though

	perl -MG=R -e "print join(qq{\n}, @ARGV)" HTTPPost\\*.txt
 or
	perl -MG=R -e "print join(qq{\n}, @ARGV)" HTTPPost/*.txt

Notice the double backslash!

=head2 Functions

The module optionaly exports these functions:

=head3 GetCommandLine

	$cmdline = GetCommandLine();

Under windows returns the raw command line that was used to start the script, under Unix
returns

	if ($0 eq '-e') {
		return "$^X -e '' " . q{'}.join(q{' '}, map {s/['\\]/\\$1/g} @ARGV).q{'};
	} else {
		return "$^X '$0' " . q{'}.join(q{' '}, map {s/['\\]/\\$1/g} @ARGV).q{'};
	}

if the module was imported from the script and

	if ($0 eq '-e') {
		return "$^X -e '' " . q{"}.join(q{" "}, map {s/["\\]/\\$1/g} @ARGV).q{"};
	} else {
		return "$^X '$0' " . q{"}.join(q{" "}, map {s/["\\]/\\$1/g} @ARGV).q{"};
	}

if it was loaded because of -MG in perl parameters or in $ENV{PERL5OPT}.
In this second case if you C<ParseCommandLine(StripPerlFromCommandLine(GetCommandLine()))>
the parameters will be globbed.

=head3 StripPerlFromCommandLine

	$params = StripPerlFromCommandLine($cmdline);

Strips perl interpreter, all its options and the script name or C<-e> and the code from
the command line. You have to use this function before you pass the command line to
ParseCommandLine().

=head3 ParseCommandLine

	@argv = ParseCommandLine($cmdline);

Parses the string you give it, globs the wildcards, executes the backticks and returns a list of parameters.

=head2 AUTHOR

Jenda@Krynicky.cz

Bill Odom (wnodom@intrasection.com)

Radomir Starostik E<lt>rsta@bach.czE<gt> (The XS part)

=cut

